## Security Analysis Mode — White‑Box Pentest Plan (OWASP ASVS L2) — Supplemental to Xray and SonarQube

Purpose: Generate a complete, code‑derived security testing plan to reach OWASP ASVS 4.0.3 Level 2. This document produces: (1) a per‑endpoint/role plan from the codebase, (2) headless ZAP automation configs for DAST, and (3) curl‑based role/ownership tests, all runnable in restricted CI/CD.

### Assumptions
- Target standard: OWASP ASVS 4.0.3 Level 2.
- Evidence required for both allow and deny paths (logs, traces, responses).

### Inputs
- Source + configs, API contracts (OpenAPI/WSDL), Dockerfiles/env, gateway/reverse‑proxy settings.
- Roles/permissions matrix (if available), core user flows, critical assets/invariants.
- Test account credentials (username/password) for a typical role to exercise flows.
- **Note**: If `openapi.yml` or `swagger.json` found in codebase, use for comprehensive endpoint discovery and parameter mapping.
 - **Note**: The curl role‑matrix harness uses the OIDC Resource Owner Password Credentials (password) grant; ensure it is enabled for this application/client in your OIDC provider for the test environment. If not permitted, replace with authorization code device/headless login and inject tokens accordingly.

### Deliverables
- Code‑derived pentest plan per component/endpoint mapped to ASVS L2.
- Executable test cases (positive/negative/ownership/tenant/concurrency) with expected outcomes.
- Prioritized findings with minimal fixes and verification steps.
- whitebox-testing.sh: generated by AI with flags:
  - --app-url, --oidc-auth-url, --oidc-token-url, --oidc-client-id, --oidc-redirect-uri, --oidc-scope, --username, --password, [--oidc-client-secret]
  - The script must export envs used by ZAP and curl harness accordingly.

### Procedure
1) Code‑Driven Discovery (White‑Box)
   - Entry points: list routes/controllers/SOAP operations, verbs, parameters, identifiers (check for openapi.yml/swagger.json for comprehensive API mapping).
   - Authentication: flows, session/token lifecycle, CSRF defenses for state changes.
   - Authorization: role/attribute checks, deny‑by‑default, ownership filters, tenant boundaries.
   - Session Management: timeout, fixation, logout, secure flags, regeneration.
   - Validation/Deserialization: schemas, canonicalization, output encoding; unsafe parsers.
   - Secrets/Crypto/TLS: key usage, algorithms/modes, rotations, transport settings.
   - Platform/Headers: CORS/CSP/HSTS, error handling/logging (no sensitive data), rate limits.
   - Debug/admin endpoints: ensure debug/management endpoints (e.g., actuator, debug flags) are disabled in production.
   - XML/SOAP: XXE off, schema validation, WS‑Security, safe XML factories.
   - Dependencies: known vulnerabilities, integrity checks, license compliance (review Xray scan results).
   - Build the Role/Permission Matrix from code references (guards, annotations, policies, queries).

2) Derive Test Plan (From Code to Runtime)
   - For each protected operation: define tests → allow, deny, ownership, cross‑tenant, mass‑assignment.
   - Add workflow abuse tests: step‑skip, replay/reorder, idempotency; race tests with concurrent calls.
   - Add injection/input tests: SQL/JPQL/NoSQL/LDAP/XPath/template; file upload traversal and content‑type.
   - Add session/CSRF/CORS/header checks; secrets exposure checks (configs/logs/images).
   - Add token refresh/rotation and expiry handling checks; verify session ID regeneration on privilege changes.
   - Add cache-control checks for sensitive responses (Cache-Control: no-store; Pragma: no-cache).
   - Add business logic tests: parameter pollution, time manipulation, state transitions, privilege escalation.
   - Add deserialization/parser tests: XML/JSON bombs, XXE, unsafe object construction.
   - Map each test to ASVS L2 control(s) (V2 AuthN, V3 Session, V4 Access, V5 Validation, V7 Errors, V9 Comms, V12 Files, V13 API).

3) Execute Against Running App (Black‑Box, Plan‑Driven)
   - Use multiple users (different roles) and tenants; capture tokens/cookies; replay with modifications.
   - Validate each matrix item; collect evidence (status, body, headers, logs); verify no data leakage.
   - Run concurrency tests on state changes; assert no duplicates/illegal transitions.

4) Rate, Fix, Verify
   - Rate severity using CVSS 3.1 (impact/exploitability, scope for authZ/data exposure).
   - Priority: Critical (9.0+), High (7.0-8.9), Medium (4.0-6.9), Low (0.1-3.9).
   - Provide minimal, actionable remediation with code examples; define verification steps.
   - Re‑test and record closure evidence mapped to ASVS L2.

### Templates

```format
Component: [file/endpoint]
Context: [function, data sensitivity, roles]
White‑Box: AuthN/AuthZ | Session | Validation | Headers/CSRF/CORS | Secrets/Crypto | XML/SOAP | Dependencies
Black‑Box: Bypass (IDOR/force‑browse/param) | Logic (skip/replay/race) | Injection/File/SSRF | Business Logic
ASVS L2: [V2.x.x, V3.x.x, V4.x.x, V5.x.x, V7.x.x, V9.x.x, V10.x.x, V12.x.x, V13.x.x]
Risk: CVSS 3.1 [score], Impact: [data/system], Likelihood: [attack complexity]
Fix: [specific code change/config]; Verify: [test steps, success criteria]
```

```format
Role Matrix Item
Operation: [resource/action]
Allowed: [roles]; Denied: [roles]
Ownership/Tenant: [rule]
Expected: [status, body]
Evidence: [code refs, logs]
```

### Outputs
- whitebox-testing.sh: orchestrator script (ZAP OIDC Auth Code + PKCE, curl role-matrix tests) for restricted CI/CD.
- whitebox-testing.md: tester guide that includes:
  - Pentest plan with per-endpoint role matrix and pass/fail results.
  - Findings with fixes and verification mapped to ASVS L2.
  - Threat model and abuse-cases summary.
  - Prerequisites and troubleshooting notes.
- ZAP reports (zap-report-oidc.html, zap-report-oidc.json) and Xray dependency scan results.

### Tooling
Selected tools suitable for restricted CI/CD.

| Tool | Primary surface | Endpoint types | Typical entrypoints | Auth with provided creds | URL |
| --- | --- | --- | --- | --- | --- |
| OWASP Zed Attack Proxy (ZAP) | UI and APIs (HTTP/S) | REST, GraphQL, SOAP, static assets | Login page, /api/**, SOAP actions | Log in via browser through proxy to capture session, or configure Context/Scripted Auth; for APIs, add Authorization header/cookie from login | https://www.zaproxy.org/ |

Notes (restricted CI/CD):
- ZAP: Use the official Docker image and the Automation Framework (headless). Disable update checks and pre-bundle required add-ons if egress is blocked. Authenticate using the provided username/password via Context login or scripts.

### Command-line tasks

#### ZAP (CI/CD headless, OIDC Authorization Code + PKCE)
```bash
# 1) Set environment (these should be passed by whitebox-testing.sh flags)
export TARGET_BASE_URL="https://app.example.local"
export OIDC_AUTH_URL="https://idp.example.com/oauth2/v2.0/authorize"   # authorization endpoint
export OIDC_TOKEN_URL="https://idp.example.com/oauth2/v2.0/token"       # token endpoint
export OIDC_CLIENT_ID="your-client-id"
export OIDC_REDIRECT_URI="http://127.0.0.1:18000/callback"             # loopback receiver in CI
export OIDC_SCOPE="openid profile email offline_access"
export LOGIN_USERNAME="user@example"
export LOGIN_PASSWORD="changeMe"

# 2) Helper to perform Auth Code + PKCE (headless) and output access_token
#    Implement this with Playwright + openid-client (or Selenium) to:
#      - Launch headless browser to $OIDC_AUTH_URL with PKCE
#      - Log in with $LOGIN_USERNAME/$LOGIN_PASSWORD
#      - Capture the code at $OIDC_REDIRECT_URI and exchange for tokens at $OIDC_TOKEN_URL
#      - Print the access_token to stdout
#    The AI should generate this script as part of whitebox-testing.sh when executed.
#    Placeholder call shown here:
ACCESS_TOKEN=$(node oidc-auth-code.js)
export ZAP_BEARER_TOKEN="$ACCESS_TOKEN"

# 3) Provide an HTTP Sender script to attach the token to all requests
cat > zap-add-bearer.js <<'EOF'
function sendingRequest(msg, initiator, helper) {
    var token = java.lang.System.getenv('ZAP_BEARER_TOKEN');
    if (token != null && token.length() > 0) {
        msg.getRequestHeader().setHeader('Authorization', 'Bearer ' + token);
    }
}
function responseReceived(msg, initiator, helper) {}
EOF

# 4) Minimal Automation config that loads the script and scans
cat > zap-automation-oidc.yaml <<'EOF'
env:
  contexts:
    - name: app
      urls:
        - "${TARGET_BASE_URL}"
  parameters:
    failOnError: true
    failOnWarning: false
jobs:
  - type: script
    parameters:
      engine: ECMAScript : Graal.js
      name: add-bearer
      type: httpsender
      file: /zap/wrk/zap-add-bearer.js
      enabled: true
  - type: spider
    parameters:
      context: app
  - type: passiveScan-config
    parameters:
      scanOnlyInScope: true
      enableTags: true
  - type: activeScan
    parameters:
      context: app
      policy: API-scan-minimal
  - type: report
    parameters:
      template: traditional-html
      reportFile: zap-report-oidc.html
      reportDir: /zap/wrk
  - type: report
    parameters:
      template: traditional-json
      reportFile: zap-report-oidc.json
      reportDir: /zap/wrk
EOF

# 5) Run ZAP with the OIDC header injection script
docker run --rm \
  -v "$PWD:/zap/wrk" \
  -u root \
  -e JAVA_OPTS="-Djava.awt.headless=true" \
  -e ZAP_BEARER_TOKEN="$ZAP_BEARER_TOKEN" \
  owasp/zap2docker-stable \
  zap.sh -cmd -silent -autorun /zap/wrk/zap-automation-oidc.yaml
```

#### Additional runtime evidence (lockout, cookies, TLS, cache-control, debug endpoints, error redaction, token refresh)
# Note: Assumes ACCESS_TOKEN and auth() are available (e.g., from the curl harness).
```bash
# Login brute-force / lockout (adjust thresholds to policy)
for i in $(seq 1 6); do
  curl -sk -o /dev/null -w "%{http_code}\n" -X POST "$TARGET_BASE_URL/login" \
    -H 'Content-Type: application/x-www-form-urlencoded' \
    --data "username=$LOGIN_USERNAME&password=WrongPass$i" || true
done
# Expect lockout or exponential backoff documented in whitebox-testing.md

# Cookie flags
curl -sk -D - "$TARGET_BASE_URL/" -o /dev/null | grep -i '^set-cookie:'
# Verify Secure; HttpOnly; SameSite in output

# TLS/HSTS/ciphers
curl -skI "$TARGET_BASE_URL/" | grep -i strict-transport-security || true
echo | openssl s_client -connect $(echo "$TARGET_BASE_URL" | sed 's#https://##;s#/$##'):443 -tls1_2 2>/dev/null | openssl x509 -noout -text | head -n 20

# Error redaction (no stack traces/PII)
curl -sk "$TARGET_BASE_URL/this-should-404" -D - -o >(head -n 40) | head -n 1

# Cache-control for sensitive responses (ASVS V8)
curl -skI -H "$(auth)" "$TARGET_BASE_URL/api/v1/profile" | egrep -i '^(cache-control|pragma):'
# Expect: Cache-Control: no-store, Pragma: no-cache

# Debug/admin endpoints disabled (ASVS V14/V10) — must not be 200 OK
for p in /actuator /actuator/env /actuator/heapdump /manage /debug /h2-console; do
  s=$(curl -sk -o /dev/null -w "%{http_code}" "$TARGET_BASE_URL$p")
  if [ "$s" = "200" ]; then echo "[FAIL] $p exposed (200)"; exit 1; else echo "[PASS] $p not publicly enabled ($s)"; fi
done

# Token refresh/rotation (ASVS V3) — requires REFRESH_TOKEN to be available
if [ -n "${REFRESH_TOKEN:-}" ]; then
  NEW_ACCESS=$(curl -s -X POST "$OIDC_TOKEN_URL" \
    -H 'Content-Type: application/x-www-form-urlencoded' \
    -d "grant_type=refresh_token&refresh_token=$REFRESH_TOKEN&client_id=$OIDC_CLIENT_ID&client_secret=$OIDC_CLIENT_SECRET&scope=$OIDC_SCOPE" \
    | jq -r '.access_token')
  if [ -n "$NEW_ACCESS" ] && [ "$NEW_ACCESS" != "$ACCESS_TOKEN" ]; then
    echo "[PASS] refresh rotated access token"
  else
    echo "[FAIL] refresh did not rotate access token"; exit 1
  fi
else
  echo "[SKIP] REFRESH_TOKEN not set; cannot verify refresh rotation"
fi

# Token expiry handling — requires short TTL or pre-expired token to assert 401
echo "[INFO] Ensure a short access token TTL in CI to assert 401 on expiry, then validate refresh path"
```


#### curl role‑matrix harness (deny/allow/ownership/tenant)
# Requires OIDC password grant (ROPC) enabled for this application/client at the OIDC provider in the test environment.
```bash
set -euo pipefail

BASE_URL="${APP_URL:-https://app.example.local}"
# Obtain a user access token via OIDC Resource Owner Password flow
export OIDC_TOKEN_URL="${OIDC_TOKEN_URL:-https://idp.example.com/oauth2/token}"
export OIDC_CLIENT_ID="${OIDC_CLIENT_ID:-your-client-id}"
export OIDC_CLIENT_SECRET="${OIDC_CLIENT_SECRET:-}"   # if required
export OIDC_SCOPE="${OIDC_SCOPE:-openid profile email}"
export OIDC_USERNAME="${USERNAME:-user@example}"
export OIDC_PASSWORD="${PASSWORD:-changeMe}"

ACCESS_TOKEN=$(curl -s -X POST "$OIDC_TOKEN_URL" \
  -H 'Content-Type: application/x-www-form-urlencoded' \
  -d "grant_type=password&username=$OIDC_USERNAME&password=$OIDC_PASSWORD&client_id=$OIDC_CLIENT_ID&client_secret=$OIDC_CLIENT_SECRET&scope=$OIDC_SCOPE" \
  | jq -r '.access_token')

auth() { echo "Authorization: Bearer $ACCESS_TOKEN"; }

assert_status() {
  local expected="$1"; shift
  local name="$1"; shift
  local status
  status=$(curl -sk -o /dev/null -w "%{http_code}" "$@")
  if [ "$status" != "$expected" ]; then
    echo "[FAIL] $name expected=$expected got=$status" >&2
    exit 1
  fi
  echo "[PASS] $name ($status)"
}

# Examples (adapt endpoints/IDs/tenants/roles):
# 1) Allow path: user with ROLE_EDITOR can update own resource
assert_status 200 "editor can update self" \
  -X PATCH "$BASE_URL/api/v1/resources/123" \
  -H "$(auth)" -H 'Content-Type: application/json' \
  --data '{"name":"new"}'

# 2) Deny path: user without role cannot update
assert_status 403 "viewer cannot update" \
  -X PATCH "$BASE_URL/api/v1/resources/123" \
  -H "$(auth)" -H 'Content-Type: application/json' \
  --data '{"name":"new"}'

# 3) Ownership: editor cannot read someone else’s object
assert_status 403 "editor blocked on foreign object" \
  -X GET "$BASE_URL/api/v1/resources/999" \
  -H "$(auth)"

# 4) Tenant isolation: cross-tenant access denied
assert_status 403 "cross-tenant blocked" \
  -X GET "$BASE_URL/api/v1/tenants/other-tenant/resources" \
  -H "$(auth)"

# 5) Mass-assignment: server ignores immutable fields
status=$(curl -sk -H "$(auth)" -H 'Content-Type: application/json' \
  -d '{"id":9999,"role":"ADMIN"}' "$BASE_URL/api/v1/resources/123" \
  -o >(tee /tmp/body.json) -w "%{http_code}")
echo "update status=$status"
jq -e '.role != "ADMIN" and .id == 123' /tmp/body.json >/dev/null || {
  echo "[FAIL] mass-assignment prevented not satisfied"; exit 1; }
echo "[PASS] mass-assignment prevented"

# 6) Concurrency/idempotency: send two PATCHes concurrently and assert single effect
tmp1=$(mktemp); tmp2=$(mktemp)
(curl -sk -X PATCH "$BASE_URL/api/v1/resources/123" -H "$(auth)" -d '{"op":"increment"}' -H 'Content-Type: application/json' -o "$tmp1" &)
(curl -sk -X PATCH "$BASE_URL/api/v1/resources/123" -H "$(auth)" -d '{"op":"increment"}' -H 'Content-Type: application/json' -o "$tmp2" &)
wait
# Fetch state and assert exactly one increment occurred (replace with real check)
final=$(curl -sk -H "$(auth)" "$BASE_URL/api/v1/resources/123")
echo "$final" | jq . > /dev/null || { echo "[WARN] add real assertion here"; }

# 7) Session security: test logout invalidation and timeout
logout_resp=$(curl -sk -X POST "$BASE_URL/logout" -H "$(auth)" -w "%{http_code}")
echo "Logout status: $logout_resp"
# Try using token after logout - should fail
assert_status 401 "token invalid after logout" \
  -X GET "$BASE_URL/api/v1/profile" -H "$(auth)"

echo "Role-matrix tests completed"
```
 
### Example Docker image with SPA/AJAX Spider support

Use a custom ZAP image that pre-installs the AJAX Spider and Selenium add-ons and includes headless browsers for JS-heavy pages.

```Dockerfile
FROM owasp/zap2docker-weekly

# Headless browsers and utilities
RUN apt-get update \
 && apt-get install -y chromium chromium-driver firefox-esr jq ca-certificates \
 && rm -rf /var/lib/apt/lists/*

# Pre-install ZAP add-ons (avoid fetching from marketplace at runtime)
RUN zap.sh -cmd -silent -addonupdate \
 && zap.sh -cmd -silent -addoninstall ajaxSpider \
 && zap.sh -cmd -silent -addoninstall selenium \
 && zap.sh -cmd -silent -addoninstall ascanrules \
 && zap.sh -cmd -silent -addoninstall pscanrulesBeta

ENV JAVA_OPTS="-Djava.awt.headless=true"

# Default workdir for reports/configs
WORKDIR /zap/wrk

# Example run (mount your zap-automation-oidc.yaml at /zap/wrk)
# docker build -t custom-zap:latest .
# docker run --rm -v "$PWD:/zap/wrk" custom-zap:latest \
#   zap.sh -cmd -silent -autorun /zap/wrk/zap-automation-oidc.yaml
```